<?php
// public/china_step4_submit.php
require_once __DIR__ . '/../config/auth.php';
require_login();
csrf_token();
require_once __DIR__ . '/../config/db.php';

if (empty($_SESSION['china_application_id'])) {
    header('Location: china_step1.php');
    exit;
}

// CSRF
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== ($_SESSION['csrf_token'] ?? '')) {
    die('Invalid CSRF token');
}

$applicationId = $_SESSION['china_application_id'];
$errors = [];

// مسیر پایه آپلود
$baseUploadDir = dirname(__DIR__) . '/uploads/china/';

// مطمئن شو پوشه وجود داره
if (!is_dir($baseUploadDir)) {
    mkdir($baseUploadDir, 0777, true);
}

// تنظیمات عمومی فایل
$maxSize = 2 * 1024 * 1024; // 2MB
$allowedImageExt = ['jpg','jpeg','png'];
$allowedPdfExt   = ['pdf'];

$filesMap = [
    'passport'   => ['required' => true,  'label' => 'اسکن پاسپورت'],
    'photo'      => ['required' => true,  'label' => 'عکس پرسنلی'],
    'ticket'     => ['required' => true,  'label' => 'بلیط رفت و برگشت'],
    'hotel'      => ['required' => true,  'label' => 'رزرو هتل'],
    'bank'       => ['required' => true,  'label' => 'تمکن مالی'],
    'invitation' => ['required' => false, 'label' => 'دعوت‌نامه'],
];

// تابع کمکی برای واترمارک (ساده؛ مبتنی بر Imagick برای تصاویر)
// اگر Imagick نصب نبود، می‌توانی فعلاً فقط move_uploaded_file انجام بدی.
function add_watermark_if_image(string $filePath): void
{
    if (!extension_loaded('imagick')) {
        // اگر Imagick فعال نیست، فعلاً هیچ کاری نکن
        return;
    }

    try {
        $im = new Imagick($filePath);
        $format = strtolower($im->getImageFormat());
        if (!in_array($format, ['jpeg','jpg','png'], true)) {
            $im->destroy();
            return;
        }

        $draw = new ImagickDraw();
        $draw->setFontSize(24);
        $draw->setFillColor(new ImagickPixel('rgba(255,255,255,0.35)'));
        $draw->setGravity(Imagick::GRAVITY_CENTER);

        $text = 'For China Visa Application – YourSite.com';

        $im->annotateImage($draw, 0, 0, 35, $text);
        $im->writeImage($filePath);
        $im->destroy();
    } catch (Exception $e) {
        // در صورت خطا، سکوت می‌کنیم تا مدارک از بین نرود
    }
}

// پردازش فایل‌ها
$pdo = get_pdo();
$pdo->beginTransaction();

try {
    foreach ($filesMap as $fieldName => $cfg) {
        $isRequired = $cfg['required'];
        $label      = $cfg['label'];

        if (!isset($_FILES[$fieldName]) || $_FILES[$fieldName]['error'] === UPLOAD_ERR_NO_FILE) {
            if ($isRequired) {
                $errors[$fieldName] = $label . ' الزامی است.';
            }
            continue;
        }

        $file = $_FILES[$fieldName];

        if ($file['error'] !== UPLOAD_ERR_OK) {
            $errors[$fieldName] = 'خطا در آپلود ' . $label . '.';
            continue;
        }

        if ($file['size'] > $maxSize) {
            $errors[$fieldName] = $label . ' نباید بیشتر از ۲ مگابایت باشد.';
            continue;
        }

        $originalName = $file['name'];
        $ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));

        if (!in_array($ext, array_merge($allowedImageExt, $allowedPdfExt), true)) {
            $errors[$fieldName] = $label . ' فقط با فرمت JPG / PNG / PDF قابل آپلود است.';
            continue;
        }

        // ساخت نام یکتا
        $newFileName = 'cn_' . $applicationId . '_' . $fieldName . '_' . time() . '.' . $ext;
        $targetPath  = $baseUploadDir . $newFileName;

        if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
            $errors[$fieldName] = 'خطا در ذخیره‌سازی فایل ' . $label . '.';
            continue;
        }

        // اگر تصویر است → واترمارک
        if (in_array($ext, $allowedImageExt, true)) {
            add_watermark_if_image($targetPath);
        }

        // ذخیره / آپدیت در جدول visa_documents
        $mimeType = mime_content_type($targetPath);

        // چک می‌کنیم آیا همین نوع مدرک قبلاً ثبت شده
        $stmt = $pdo->prepare("
            SELECT id FROM visa_documents
            WHERE application_id = ? AND doc_type = ?
        ");
        $stmt->execute([$applicationId, $fieldName]);
        $docId = $stmt->fetchColumn();

        if ($docId) {
            // آپدیت
            $stmt = $pdo->prepare("
                UPDATE visa_documents
                SET file_path = ?, original_name = ?, mime_type = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$newFileName, $originalName, $mimeType, $docId]);
        } else {
            // اینسرت
            $stmt = $pdo->prepare("
                INSERT INTO visa_documents
                    (application_id, doc_type, file_path, original_name, mime_type, created_at, updated_at)
                VALUES (?,?,?,?,?,NOW(),NOW())
            ");
            $stmt->execute([$applicationId, $fieldName, $newFileName, $originalName, $mimeType]);
        }
    }

    if ($errors) {
        $pdo->rollBack();
        $_SESSION['form_errors'] = $errors;
        header('Location: china_step4.php');
        exit;
    }

    // همه اوکی → مرحله را آپدیت کنیم
    $stmt = $pdo->prepare("UPDATE visa_applications SET current_step = 4, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$applicationId]);

    $pdo->commit();
} catch (Exception $e) {
    $pdo->rollBack();
    die('Upload error: ' . $e->getMessage());
}
$_SESSION['success_message'] = 'مدارک با موفقیت ذخیره شدند.';
header('Location: china_step5.php');
exit;
