<?php
// includes/otp.php

require_once __DIR__ . '/../config/db.php';

function otp_table_name(): string {
    return 'visa_login_otps';
}


function otp_normalize_digits(string $s): string {
    $map = [
        '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9',
        '٠'=>'0','١'=>'1','٢'=>'2','٣'=>'3','٤'=>'4','٥'=>'5','٦'=>'6','٧'=>'7','٨'=>'8','٩'=>'9',
    ];
    return strtr(trim($s), $map);
}


function otp_issue(string $email): array
{
    $pdo = get_pdo();
    $email = strtolower(trim($email));

    // rate limit: حداقل 60 ثانیه بین درخواست‌ها
    try {
        $stmt = $pdo->prepare("SELECT created_at FROM " . otp_table_name() . " WHERE email = ? ORDER BY id DESC LIMIT 1");
        $stmt->execute([$email]);
        $last = $stmt->fetchColumn();
        if ($last) {
            $lastTs = strtotime($last);
            if ($lastTs && (time() - $lastTs) < 60) {
                return ['ok' => false, 'error' => 'لطفاً 1 دقیقه دیگر دوباره تلاش کنید.'];
            }
        }
    } catch (Throwable $e) {
        // اگر جدول نبود یا خطا داشت، پایین‌تر پیام می‌دهیم
    }

    $code = (string)random_int(100000, 999999);
    $hash = password_hash($code, PASSWORD_DEFAULT);
    $expiresAt = (new DateTime('+10 minutes'))->format('Y-m-d H:i:s');
    $now = (new DateTime())->format('Y-m-d H:i:s');

    try {
        $stmt = $pdo->prepare("INSERT INTO " . otp_table_name() . " (email, code_hash, expires_at, created_at) VALUES (?, ?, ?, ?)");
        $stmt->execute([$email, $hash, $expiresAt, $now]);
    } catch (Throwable $e) {
        return ['ok' => false, 'error' => 'جدول OTP در دیتابیس پیدا نشد یا ساختارش درست نیست.'];
    }

    return ['ok' => true, 'code' => $code, 'expires_at' => $expiresAt];
}

function otp_verify(string $email, string $code): array
{
    $pdo = get_pdo();
    $email = strtolower(trim($email));
    $code = otp_normalize_digits($code);

    try {
        $stmt = $pdo->prepare("
            SELECT id, code_hash, expires_at, used_at
            FROM " . otp_table_name() . "
            WHERE email = ?
            ORDER BY id DESC
            LIMIT 1
        ");
        $stmt->execute([$email]);
        $row = $stmt->fetch();
    } catch (Throwable $e) {
        return ['ok' => false, 'error' => 'OTP روی سرور فعال نیست (جدول/کد).'];
    }

    if (!$row) return ['ok' => false, 'error' => 'ابتدا کد را درخواست کنید.'];
    if (!empty($row['used_at'])) return ['ok' => false, 'error' => 'این کد قبلاً استفاده شده است.'];

    $exp = strtotime($row['expires_at']);
    if ($exp && time() > $exp) return ['ok' => false, 'error' => 'کد منقضی شده است. دوباره درخواست دهید.'];

    if (!password_verify($code, $row['code_hash'])) {
        return ['ok' => false, 'error' => 'کد وارد شده صحیح نیست.'];
    }

    // mark used
    $stmt = $pdo->prepare("UPDATE " . otp_table_name() . " SET used_at = NOW() WHERE id = ?");
    $stmt->execute([$row['id']]);

    return ['ok' => true];
}
